/****************************************************************************
*
*    Copyright (c) 2005 - 2014 by Vivante Corp.  All rights reserved.
*
*    The material in this file is confidential and contains trade secrets
*    of Vivante Corporation. This is proprietary information owned by
*    Vivante Corporation. No part of this work may be disclosed,
*    reproduced, copied, transmitted, or used in any way for any purpose,
*    without the express written permission of Vivante Corporation.
*
*****************************************************************************/


/*!
@header tinyUI kernel porting layer
A number of functions that need to be implemented if the tinyUI driver is ported to a new platform.

@copyright Vivante Corporation
*/

#ifndef _tiny_ui_hal_h_
#define _tiny_ui_hal_h_

#include "tiny_ui_platform.h"

#ifdef __cplusplus
extern "C" {
#endif

/*!
@brief Allocate memory.

@discussion
The kernel driver needs to allocate memory from the system heap for its internal structures. This is the wrapper function to
allocate the memory.

@param size
The number of bytes to allocate.

@result
A pointer to the allocated memory or <code>NULL</code> if there is not enough memory.
*/
void * tiny_ui_hal_alloc(unsigned long size);

/*!
@brief Free memory.

@discussion
Free the memory allocated previously by {@link tiny_ui_hal_alloc}.

@param memory
A pointer to the memory area the got previously allocated by {@linkn tiny_ui_hal_alloc}.
*/
void tiny_ui_hal_free(void * memory);

/*!
 @brief Wait a number of milliseconds.

 @discussion
 The tinyUI hardware requires some waiting when changing clock frequencies or issuing a reset. This is the wrapper function
 for the delay function.

 @param milliseconds
 The number of milliseconds to wait.
 */
void tiny_ui_hal_delay(uint32_t milliseconds);

/*!
 @brief Initialize the hardware.

 @discussion
 The tinyUI kernel knows how to program its own hardware, but in any SOC there might be additional control required for
 turning on the power or initializing the clocks. This function gets called by the tinyUI kernel before the tinyUI graphics
 hardware gets initialized by the tinyUI kernel itself and allows for SOC power management control.

 The implementer should make sure that on exit of this function the power and clock to the tinyUI graphics hardware is
 turned on and stable.
 */
void tiny_ui_hal_initialize(void);

/*!
 @brief Uninitialize the hardware.

 @discussion
 The tinyUI kernel knows how to program its own hardware, but in any SOC there might be additional control required for
 turning off the power or uninitializing the clocks. This function gets called by the tinyUI kernel after the tinyUI
 graphics hardware gets uninitialized by the tinyUI kernel itself and allows for SOC power management control.

 On exit of this function it is okay to have the power and/or clock to the tinyUI graphics hardware turned off.
 */
void tiny_ui_hal_deinitialize(void);

/*!
 @brief Allocate contiguous video memory.

 @discussion
 Any memory the tinyUI graphics hardware will see should be allocated as contiguous memory. Any allocated memory will be
 addressed through an opaque handle, usually a pointer to an opaque structure. The porting layer can put any information it
 needs inside this structure.

 @param size
 The number of bytes to allocate.

 @param logical
 A pointer to a variable that will receive the logical address of the allocated memory for the CPU.

 @param gpu
 A pointer to a variable that will receive the physical address of the allocated memory for the tinyUI graphics hardware.

 @result
 A pointer to an opaque structure that will be used as the memory handle. <code>NULL</code> should be returned if there is not
 enough memory.
 */
void * tiny_ui_hal_allocate_contiguous(unsigned long size, void ** logical, uint32_t * gpu);

/*!
 @brief Free contiguous video memory.

 @discussion
 Free the memory allocated by {@link tiny_ui_hal_allocate_contiguous}. After this function returns, the associated memory
 handle is no longer a valid handle.

 @param memory_handle
 A pointer to an opaque structure returned by {@link tiny_ui_hal_allocate_contiguous}.
 */
void tiny_ui_hal_free_contiguous(void * memory_handle);

/*!
 @brief Map contiguous logical or physical memory into the tinyUI graphics hardware space.

 @discussion
 Any memory, like a frame buffer or some pre-allocated image or path data, needs to be mapped into the tinyUI graphics
 hardware address space and wrapped by a memory handle. This allows the tinyUI graphics hardware access that memory
 directly.

 Either a logical or a physical address should be passed in to map.

 @param size
 The number of bytes to map.

 @param logical
 The logical address of the memory region to map or <code>NULL</code> if the logical address is not known.

 @param physical
 The physical address of the memory region to map if <code>logical</code> is <code>NULL</code>.

 @param gpu
 A pointer to a variable that will receive the tinyUI graphics hardware addressable address of the mapped region.

 @result
 A pointer to an opaque structure that will be used as the memory handle. <code>NULL</code> should be returned if there is
 not enough system resources to map the region.
 */
void * tiny_ui_hal_map(unsigned long size, void * logical, uint32_t physical, uint32_t * gpu);

/*!
 @brief Unmap a previously mapped region.

 @discussion
 If a mapped region by {@link tiny_ui_hal_map} is no longer needed, it should be unmapped to free up any allocated system
 resources used when mapping the region.

 @param memory_handle
 A pointer to an opaque structure returned by {@link tiny_ui_hal_map}.
 */
void tiny_ui_hal_unmap(void * memory_handle);

/*!
 @brief Execute a memory barrier.

 @discussion
 Some systems require a a memory barrier to make sure all store operations in the CPU have been handled. This is the wrapper
 function for a memory barrier.
 */
void tiny_ui_hal_barrier(void);

/*!
 @brief Read data from a register from the tinyUI graphics hardware.

 @discussion
 In order to communicate with the tinyUI graphics hardware, the kernel needs to read and write to some hardware registers.
 In each SOC those registers could be allocated at a different space in the physical memory map.

 @param address
 The relative address of the tinyUI graphics hardware register to read from.

 @result
 The 32-bit value returned from reading the register.
 */
uint32_t tiny_ui_hal_peek(uint32_t address);

/*!
 @brief Write data to a register from the tinyUI graphics hardware.

 @discussion
 In order to communicate with the tinyUI graphics hardware, the kernel needs to read and write to some hardware registers.
 In each SOC those registers could be allocated at a different space in the physical memory map.

 @param address
 The relative address of the tinyUI graphics hardware register to write to.

 @param data
 The data to write to the tinyUI graphics hardware register.
 */
void tiny_ui_hal_poke(uint32_t address, uint32_t data);

/*!
 @brief Wait until an interrupt from the tinyUI graphics hardware has been received.

 @discussion
 Currently, the tinyUI API is synchronous. This means that after each call it will wait until the tinyUI graphics hardware
 has completed. The tinyUI graphics hardware will send an interrupt when it is finished, and this function will wait until
 that interrupt has been received by the operating system.

 A timeout value is specified in order if the kernel wants to wait for a specific number of milliseconds fir the interrupt to
 occur. If the interrupt does not occur in the specified timeout, a timeout error will be returned.

 @param timeout
 The number of milliseconds to wait for the interrupt before returning a timeout error. If <code>timeout = 0xFFFFFFFF</code>
 then {@link tiny_ui_hal_wait_interrupt} will wait forever for the interrupt.

 @result
 A boolean value indicating whether the interrupt was received (1) or not (0).
 */
int32_t tiny_ui_hal_wait_interrupt(uint32_t timeout);

#ifdef __cplusplus
}
#endif
#endif // _tiny_ui_hal_h_
